package ga.view.streaming;

import ga.core.GA;
import ga.core.algorithm.interactive.ISIGA;
import ga.core.evaluation.EvaluationListener;
import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.core.individual.IIndividual;
import ga.view.appstate.SceneState;
import ga.view.appstate.menu.IMenuListenerParent;
import ga.view.appstate.menu.MenuListener;
import ga.view.interfaces.IPhenotypeGenerator;
import ga.view.interfaces.MouseListener;
import ga.view.streaming.nodes.AnchorNode;
import ga.view.streaming.nodes.EvaluationNode;
import ga.view.streaming.nodes.PanelNode;
import ga.view.streaming.nodes.PanelNode.InfoStringType;
import ga.view.streaming.nodes.PanelNodeListener;
import ga.view.streaming.showroom.CameraSettings;
import ga.view.streaming.showroom.ShowRoom;
import ga.view.streaming.showroom.ShowRoomFactory;
import ga.view.streaming.showroom.ShowRoomSettings;
import ga.view.streaming.showroom.ShowRoomSettings.ShowRoomType;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.jme3.app.Application;
import com.jme3.app.state.AppStateManager;
import com.jme3.bullet.BulletAppState;
import com.jme3.bullet.PhysicsSpace;
import com.jme3.bullet.collision.PhysicsCollisionObject;
import com.jme3.bullet.collision.shapes.BoxCollisionShape;
import com.jme3.bullet.control.RigidBodyControl;
import com.jme3.bullet.joints.Point2PointJoint;
import com.jme3.collision.CollisionResult;
import com.jme3.collision.CollisionResults;
import com.jme3.input.KeyInput;
import com.jme3.input.controls.ActionListener;
import com.jme3.input.controls.KeyTrigger;
import com.jme3.input.controls.MouseButtonTrigger;
import com.jme3.light.AmbientLight;
import com.jme3.light.DirectionalLight;
import com.jme3.material.Material;
import com.jme3.material.RenderState.BlendMode;
import com.jme3.math.ColorRGBA;
import com.jme3.math.FastMath;
import com.jme3.math.Ray;
import com.jme3.math.Vector3f;
import com.jme3.renderer.queue.RenderQueue.Bucket;
import com.jme3.renderer.queue.RenderQueue.ShadowMode;
import com.jme3.scene.Geometry;
import com.jme3.scene.Node;
import com.jme3.scene.Spatial;
import com.jme3.scene.Spatial.CullHint;
import com.jme3.scene.shape.Box;
import com.jme3.system.AppSettings;

/**
 * This interactive evaluator provides a stream of pictures of the show room.
 * There are 2 zones to upvote/downvote the individuals.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("unchecked")
public class StreamingEvaluationState<T extends IIndividual<T>> extends
    SceneState implements IInteractiveFitnessEvaluator<T>, IMenuListenerParent {
  public static final float SCENE_WIDTH = 9f;
  private static final float PANEL_SPACING = 4f;

  // the logger for this class
  private static final Logger LOGGER = Logger
      .getLogger(StreamingEvaluationState.class.getName());

  private Geometry pointer;
  private boolean mouseDown;
  private RigidBodyControl pointerControl;
  private Node panelsNode;

  private final IPhenotypeGenerator<T, ? extends Spatial> phenotypeGenerator;
  private ShowRoomFactory showRoomFactory;

  private final ShowRoomSettings srSettings;

  private final BulletAppState bulletAppState;
  private ShowRoomState<T> showRoomState;

  private EvaluationNode lowerBox, upperBox;

  private final List<AnchorNode<T>> anchorNodes = new ArrayList<AnchorNode<T>>();

  private final List<EvaluationListener<T>> listeners = new ArrayList<EvaluationListener<T>>();
  private final DragListener dragListener = new DragListener();

  private CameraSettings camSettings;

  private MenuListener menuListener;

  private AppSettings settings;

  private float speed = 1f;

  // this is overridden, because the super setEnabled() makes problems
  private boolean enabled = true;

  private ISIGA<T> algorithm;

  private Application app;

  /**
   * Instantiates a new streaming evaluation state.
   * 
   * @param phenotypeGenerator
   *          the phenotype generator
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public StreamingEvaluationState(
      final IPhenotypeGenerator<T, ? extends Spatial> phenotypeGenerator) {
    this.phenotypeGenerator = phenotypeGenerator;
    this.srSettings = new ShowRoomSettings();

    srSettings.put(ShowRoomSettings.TYPE, ShowRoomType.BOX);
    srSettings.put(ShowRoomSettings.BOX_WIDTH, 6f);
    srSettings.put(ShowRoomSettings.BOX_LENGTH, 4f);
    srSettings.put(ShowRoomSettings.BOX_HEIGHT, 2.6f);

    bulletAppState = new BulletAppState();
    bulletAppState.setThreadingType(BulletAppState.ThreadingType.PARALLEL);

  }

  @Override
  public void setMenuListener(final MenuListener menuListener) {
    this.menuListener = menuListener;
  }

  @Override
  public void initialize(final AppStateManager stateManager,
      final Application app) {
    super.initialize(stateManager, app);

    this.app = app;

    phenotypeGenerator.setAssetManager(assetManager);

    this.settings = app.getContext().getSettings();

    showRoomFactory = new ShowRoomFactory(assetManager, settings, srSettings);
    final ShowRoom showRoom = showRoomFactory.createShowRoom();

    camSettings = CameraSettings.getOrthographicSettings(showRoom, settings);

    showRoomState = new ShowRoomState<T>(this, camSettings);

    algorithm.getContext().put(GA.KEY_VALIDATION_SPACE, showRoom);

    // TODO these are ga settings and should be put to the config file
    algorithm.getContext().put(GA.KEY_GENOME_MIN_LENGTH, 1);
    algorithm.getContext().put(GA.KEY_GENOME_MAX_LENGTH, 3);
    algorithm.init();

    stateManager.attach(bulletAppState);
    bulletAppState.getPhysicsSpace().setAccuracy(0.005f);

    panelsNode = new Node("Panels");
    final DirectionalLight dl = new DirectionalLight();
    dl.setDirection(Vector3f.UNIT_Y.negate());
    panelsNode.addLight(dl);

    rootNode.addLight(new AmbientLight());

    showRoomState.initialize(stateManager, app);
    showRoomState.setEnabled(false);

    rootNode.attachChild(panelsNode);

    initCam();
    initTable();
    initPointer();

    // add the drag mapping
    // dragListener = new DragListener();
    inputManager.addMapping("drag", new MouseButtonTrigger(0));
    inputManager.addListener(dragListener, "drag");

    inputManager.addMapping("switch view", new KeyTrigger(KeyInput.KEY_V));
    inputManager.addListener(new ViewSwitchListener(), "switch view");
  }

  @Override
  public void cleanup() {
    if (algorithm != null) {
      algorithm.exit();
    }

    if (showRoomState != null) {
      showRoomState.setEnabled(false);
      stateManager.detach(showRoomState);
    }

    try {
      stateManager.detach(bulletAppState);
    } catch (final Exception e) {
      LOGGER.warning(e.toString());
    }

    super.cleanup();
  }

  @Override
  public void setEnabled(final boolean enabled) {
    // this makes problems
    // super.setEnabled(active);
    this.enabled = enabled;

    // TODO FIX the problems
    bulletAppState.setEnabled(enabled);

    dragListener.setEnabled(enabled);

    if (viewPort != null) {
      if (enabled) {
        if (!renderManager.getMainViews().contains(viewPort)) {
          viewPort = renderManager.createMainView("Scene", cam);
          viewPort.setClearFlags(true, true, true);
          viewPort.attachScene(rootNode);
        }
      } else {
        // DO NOT REMOVE PROCESSORS HERE
        // while (viewPort.getProcessors().size() > 0) {
        // SceneProcessor proc = viewPort.getProcessors().get(0);
        // viewPort.removeProcessor(proc);
        // }

        renderManager.removeMainView(viewPort);
      }
    }
  }

  // @Override
  // public boolean isEnabled() {
  // return enabled;
  // }

  @Override
  public void update(final float tpf) {
    if (!isEnabled()) {
      return;
    }

    super.update(tpf);

    // move the anchor nodes
    if (!mouseDown) {
      upperBox.setMouseOver(false);
      lowerBox.setMouseOver(false);

      if (timer.getTimeInSeconds() > 2f) {
        if (anchorNodes.size() < 5) {
          fireNewIndividualRequested();
        }

        AnchorNode<T> anchorNode = anchorNodes.get(0);
        Vector3f trans = anchorNode.getLocalTranslation();
        trans.x += tpf * speed;
        anchorNode.setLocalTranslation(trans);

        for (int i = 1; i < anchorNodes.size(); i++) {
          anchorNode = anchorNodes.get(i);

          trans = trans.add(-PANEL_SPACING, 0f, 0f);
          anchorNode.setLocalTranslation(trans);
        }

        anchorNode = anchorNodes.get(0);
        if (anchorNode.getLocalTranslation().x > SCENE_WIDTH) {
          removeAnchorNode(anchorNode);
        }
      }
    }

    if (mouseDown) {
      final CollisionResults results = findPick(rootNode);

      if (results.size() > 0) {

        final Iterator<CollisionResult> it = results.iterator();

        boolean upperBoxFound = false;
        boolean lowerBoxFound = false;

        while (it.hasNext()) {
          final CollisionResult r = it.next();

          // upperBoxFound |= r.getGeometry().getParent() == upperBox;
          // lowerBoxFound |= r.getGeometry().getParent() == lowerBox;

          if (r.getGeometry().getParent() != null
              && r.getGeometry().getParent() instanceof PanelNode) {
            final PanelNode<T> n = (PanelNode<T>) r.getGeometry().getParent();
            if (upperBox.isMouseOver() || lowerBox.isMouseOver()) {
              n.setFadeToAlpha(0.6f);
            } else {
              n.setFadeToAlpha(1.0f);
            }

            break;
          }
        }

        final Vector3f origin = cam.getWorldCoordinates(
            inputManager.getCursorPosition(), 0.0f);

        LOGGER.info(String.valueOf(origin));

        if (origin.z <= -0.015f) {
          upperBoxFound = true;
        } else if (origin.z >= 0.015f) {
          lowerBoxFound = true;
        }

        if (upperBoxFound) {
          upperBox.setMouseOver(true);
        } else {
          upperBox.setMouseOver(false);
        }

        if (lowerBoxFound) {
          lowerBox.setMouseOver(true);
        } else {
          lowerBox.setMouseOver(false);
        }

        final CollisionResult closest = results.getClosestCollision();
        final Vector3f markPos = closest.getContactPoint().setY(3f);
        pointer.setLocalTranslation(markPos);
      }
    }

    super.update(tpf);
  }

  /**
   * Inits the cam.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void initCam() {
    /** Set up camera */
    cam.setLocation(new Vector3f(0f, 10f, 0f));
    cam.lookAt(Vector3f.ZERO, Vector3f.UNIT_Z.negate());
    cam.setFrustumFar(50);

  }

  /** Make a solid floor and add it to the scene. */
  private void initTable() {
    Geometry geo = new Geometry("upperbox", new Box(.5f, 0f, .5f));

    Material mat = new Material(assetManager,
        "Common/MatDefs/Misc/Unshaded.j3md");
    mat.setTexture("ColorMap",
        assetManager.loadTexture("ga/view/resource/plus.png"));
    mat.getAdditionalRenderState().setBlendMode(BlendMode.Alpha);

    geo.setMaterial(mat);
    geo.setShadowMode(ShadowMode.Off);
    geo.rotate(0, FastMath.DEG_TO_RAD * 90f, 0f);
    geo.setLocalTranslation(0f, 0.4f, -3f);

    upperBox = new EvaluationNode("upperboxnode", geo);
    this.rootNode.attachChild(upperBox);

    geo = new Geometry("lowerbox", new Box(.5f, 0f, .5f));
    mat = new Material(assetManager, "Common/MatDefs/Misc/Unshaded.j3md");
    mat.setTexture("ColorMap",
        assetManager.loadTexture("ga/view/resource/minus.png"));
    mat.getAdditionalRenderState().setBlendMode(BlendMode.Alpha);
    geo.setMaterial(mat);
    geo.setShadowMode(ShadowMode.Off);
    geo.rotate(0f, FastMath.DEG_TO_RAD * 90f, 0f);
    geo.setLocalTranslation(0f, 0.4f, 3f);

    lowerBox = new EvaluationNode("lowerboxnode", geo);
    this.rootNode.attachChild(lowerBox);
  }

  /**
   * Inits the pointer. A red ball that marks the last spot that was "hit" by
   * the "shot".
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void initPointer() {
    final Box b = new Box(0.1f, 0.0f, 0.1f);
    pointer = new Geometry("pointer", b);
    Material pointerMat = new Material(assetManager,
        "Common/MatDefs/Misc/Unshaded.j3md");
    pointerMat.setColor("Color", new ColorRGBA(1f, 0f, 0f, 1f));

    pointerMat = new Material(assetManager, "Common/MatDefs/Misc/Unshaded.j3md");
    pointerMat.setTexture("ColorMap",
        assetManager.loadTexture("ga/view/resource/pointer.png"));
    pointerMat.getAdditionalRenderState().setBlendMode(BlendMode.Alpha);
    pointerMat.setTransparent(true);

    pointer.setMaterial(pointerMat);
    pointer.setShadowMode(ShadowMode.Off);
    pointer.setName("Pointer Node");
    pointer.setQueueBucket(Bucket.Translucent);
    pointer.setCullHint(CullHint.Always);
    pointerControl = new RigidBodyControl(new BoxCollisionShape(new Vector3f(
        .1f, .1f, .1f)), // collision
        // shape
        5f); // mass
    pointer.addControl(pointerControl);

    pointerControl.setKinematic(true);
    pointerControl
        .setCollisionGroup(PhysicsCollisionObject.COLLISION_GROUP_NONE);
    pointerControl
        .setCollideWithGroups(PhysicsCollisionObject.COLLISION_GROUP_NONE);

  }

  /**
   * Finds spatials in the scene that has been clicked.
   * 
   * @param node
   *          The parent node to check for clicks.
   * @return The results of the click.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private CollisionResults findPick(final Node node) {
    final Vector3f origin = cam.getWorldCoordinates(
        inputManager.getCursorPosition(), 0.0f);
    final Vector3f direction = cam.getWorldCoordinates(
        inputManager.getCursorPosition(), 0.3f);
    direction.subtractLocal(origin).normalizeLocal();

    final Ray ray = new Ray(origin, direction);
    final CollisionResults results = new CollisionResults();
    node.collideWith(ray, results);
    return results;
  }

  /**
   * Gets the physics space.
   * 
   * @return the physics space
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private PhysicsSpace getPhysicsSpace() {
    return bulletAppState.getPhysicsSpace();
  }

  @Override
  public void setAlgorithm(final ISIGA<T> algorithm) {
    this.algorithm = algorithm;
  }

  @Override
  public ISIGA<T> getAlgorithm() {
    return algorithm;
  }

  /**
   * Shows a panel node in show room.
   * 
   * @param panelNode
   *          the panel node
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void showInShowRoom(final PanelNode<T> panelNode) {
    LOGGER.info("SHOW IN SHOWROOM");
    showRoomState.setPanelNode(panelNode);
    this.setEnabled(false);
    speed = 1f;
  }

  /**
   * Creates a panel for an individual.
   * 
   * @param individual
   *          the individual
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void createPanel(final T individual) {
    final Spatial phenotype = phenotypeGenerator.createPhenotype(individual);

    float x;
    if (anchorNodes.size() == 0) {
      x = -SCENE_WIDTH;
    } else {
      x = anchorNodes.get(0).getLocalTranslation().x
          - (PANEL_SPACING * anchorNodes.size());
    }

    final ShowRoom showRoom = showRoomFactory.createShowRoom();
    showRoom.setPhenotype(phenotype);

    final PanelNode<T> panelNode = new PanelNode<T>(assetManager, settings,
        showRoom, camSettings, individual);

    panelNode.setInfoStringType(InfoStringType.COSTS);

    viewPort.addProcessor(panelNode.getProcessor());

    panelNode.setPanelNodeListener(new PanelNodeListener<T>() {
      @Override
      public void panelReadyToDestroy(final PanelNode<T> panelNode) {

        app.enqueue(new Callable<Void>() {
          @Override
          public Void call() throws Exception {
            removeAnchorNode(panelNode.getAnchor());
            return null;
          }
        });

      }

      @Override
      public void panelInInspectPosition(final PanelNode<T> panelNode) {
        showInShowRoom(panelNode);
        panelNode.inspectDone();
      }
    });

    panelNode.setLocalTranslation(x, 0f, 0f);

    final AnchorNode<T> anchorNode = new AnchorNode<T>(getPhysicsSpace());
    anchorNode.setLocalTranslation(x, 0f, 0f);

    this.rootNode.attachChild(anchorNode);
    this.getPhysicsSpace().add(anchorNode.getControl());

    anchorNodes.add(anchorNode);

    this.panelsNode.attachChild(panelNode);
    getPhysicsSpace().add(panelNode);

    anchorNode.attachJoint(panelNode);
  }

  /**
   * This is called when the anchorNode goes out of the screen without being
   * evaluated.
   * 
   * @param anchorNode
   *          The anchor node that leaves the screen
   */
  private void removeAnchorNode(final AnchorNode<T> anchorNode) {
    final PanelNode<T> panelNode = anchorNode.getAttachedNode();
    rootNode.detachChild(anchorNode);
    panelsNode.detachChild(panelNode);
    getPhysicsSpace().remove(anchorNode);
    getPhysicsSpace().remove(panelNode);

    anchorNode.detachJoint();
    // panelNode.getControl().destroy();
    // anchorNode.getControl().destroy();

    viewPort.removeProcessor(panelNode.getProcessor());

    anchorNodes.remove(anchorNode);
  }

  /**
   * Removes a panel from the scene.
   * 
   * @param panelNode
   *          the panel node
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void removePanel(final PanelNode<T> panelNode) {
    final AnchorNode<T> anchorNode = panelNode.getAnchor();

    // if the first anchor is removed, the second anchor must be set to the
    // location of the first
    final int index = anchorNodes.indexOf(anchorNode);
    if (anchorNodes.size() > 1 && index == 0) {
      anchorNodes.get(1).setLocalTranslation(anchorNode.getLocalTranslation());
    }

    rootNode.detachChild(anchorNode);
    getPhysicsSpace().remove(anchorNode);
    anchorNodes.remove(anchorNode);

    anchorNode.detachJoint();

    panelNode.getControl().setGravity(
        new Vector3f(0f, -90f, panelNode.getLocalTranslation().z * 20f));

    panelNode.setFadeToAlpha(0f);
  }

  @Override
  public void fireNewIndividualRequested() {
    for (final EvaluationListener<T> l : listeners) {
      l.newIndividualRequested();
    }
  }

  @Override
  public void fireIndividualEvaluated(final T individual) {
    for (final EvaluationListener<T> l : listeners) {
      l.individualEvaluated(individual);
    }
  }

  @Override
  public void addEvaluationListener(final EvaluationListener<T> listener) {
    listeners.add(listener);
  }

  @Override
  public void removeEvaluationListener(final EvaluationListener<T> listener) {
    listeners.remove(listener);
  }

  @Override
  public void evaluate(final T individual) {
    createPanel(individual);
  }

  /**
   * This is a mouse listener that detects drag-and-drop.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class DragListener extends MouseListener {
    private Point2PointJoint mouseSliderJoint;
    private PanelNode<T> selectedPanelNode;

    @Override
    public void onAction(final String name, final boolean keyPressed,
        final boolean isDoubleClick, final float tpf) {
      LOGGER.info(name + " " + keyPressed + " " + isDoubleClick);
      if (keyPressed) {
        final CollisionResults results = findPick(panelsNode);

        if (results.size() > 0) {
          mouseDown = true;
          final CollisionResult closest = results.getClosestCollision();
          final Node n = closest.getGeometry().getParent();
          if (n != null && n instanceof PanelNode<?>) {
            selectedPanelNode = (PanelNode<T>) n;

            if (isDoubleClick) {
              // if a panel is double clicked

              if (mouseSliderJoint != null) {
                getPhysicsSpace().remove(mouseSliderJoint);
                mouseSliderJoint.destroy();
                mouseSliderJoint = null;
              }

              dragListener.setEnabled(false);
              selectedPanelNode.inspect();
              speed = 0.1f;

            } else {
              // if a panel is single clicked (drag and drop)

              getPhysicsSpace().add(pointerControl);
              rootNode.attachChild(pointer);

              if (mouseSliderJoint == null) {
                mouseSliderJoint = new Point2PointJoint(pointerControl,
                    selectedPanelNode.getControl(), Vector3f.ZERO,
                    Vector3f.ZERO);
                mouseSliderJoint.setCollisionBetweenLinkedBodys(false);
                mouseSliderJoint.setDamping(1f);
                mouseSliderJoint.setImpulseClamp(150f);
                mouseSliderJoint.setTau(3f);

                getPhysicsSpace().add(mouseSliderJoint);

              }
            }
          }
        }
      } else {
        mouseDown = false;
        if (mouseSliderJoint != null) {
          getPhysicsSpace().remove(mouseSliderJoint);
          mouseSliderJoint.destroy();
          mouseSliderJoint = null;
        }

        getPhysicsSpace().remove(pointerControl);
        rootNode.detachChild(pointer);

        try {
          if (findPick(lowerBox).size() > 0) {
            // if the panel is dropped on the minus

            removePanel(selectedPanelNode);

            // decrease the fitness
            final T individual = selectedPanelNode.getIndividual();
            if (individual != null) {
              individual.setFitness(individual.getFitness() - 1);
              fireIndividualEvaluated(individual);
            }

          } else if (findPick(upperBox).size() > 0) {
            // if the panel is dropped on the plus

            removePanel(selectedPanelNode);

            final T individual = selectedPanelNode.getIndividual();
            if (individual != null) {
              individual.setFitness(individual.getFitness() + 1);
              fireIndividualEvaluated(individual);
            }
          } else {
            // if the panel is simply released

            if (selectedPanelNode != null) {
              selectedPanelNode.setFadeToAlpha(1.0f);
            }
          }

        } catch (final Exception e) {
          LOGGER.log(Level.WARNING, "An error occured", e);
        }

        selectedPanelNode = null;
      }
    }
  }

  /**
   * Keyboard listener to switch the perspective.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ViewSwitchListener implements ActionListener {
    private int i = 1;

    @Override
    public void onAction(final String name, final boolean keyPressed,
        final float tpf) {
      if (!keyPressed) {
        final int length = CameraSettings.Type.values().length;

        final CameraSettings.Type type = CameraSettings.Type.values()[++i
            % length];

        camSettings = CameraSettings.getSettings(type,
            showRoomFactory.createShowRoom(), settings);

        for (final AnchorNode<T> an : anchorNodes) {
          final PanelNode<T> pn = an.getAttachedNode();

          if (pn != null) {
            pn.setCameraSettings(camSettings);
          }
        }

        showRoomState.setCamSettings(camSettings);

        LOGGER.info("Camera is now " + type);
      }
    }
  }

}
